<?php
require __DIR__ . '/../includes/config.php';
require __DIR__ . '/../includes/functions.php';

// Simple access control
if (empty($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'admin') {
    header("Location: ../login.php");
    exit;
}

// Allowed statuses
$artistStatuses = [
    'pending'   => 'Pending',
    'on_hold'   => 'On Hold',
    'approved'  => 'Approved',
    'featured'  => 'Featured',
    'expired'   => 'Expired',
    'cancelled' => 'Cancelled',
];

$invoiceStatuses = [
    'pending'   => 'Pending',
    'paid'      => 'Paid',
    'cancelled' => 'Cancelled',
    'refunded'  => 'Refunded',
];

$userStatuses = [
    'pending'   => 'Pending',
    'active'    => 'Active',
    'suspended' => 'Suspended',
];

$notice = '';
$tab    = $_GET['tab'] ?? 'artists';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // Users
    if ($action === 'update_user') {
        $userId     = (int)($_POST['user_id'] ?? 0);
        $newStatus  = $_POST['status'] ?? '';
        $newRole    = $_POST['role'] ?? '';

        $users = read_json('users.json');
        $changed = false;

        foreach ($users as &$u) {
            if ($u['id'] === $userId) {
                if (isset($userStatuses[$newStatus])) {
                    $u['status'] = $newStatus;
                }
                if (in_array($newRole, ['artist', 'admin'], true)) {
                    $u['role'] = $newRole;
                }
                $changed = true;
                break;
            }
        }
        unset($u);

        if ($changed) {
            write_json('users.json', $users);
            $notice = "User #{$userId} updated.";
        } else {
            $notice = "User not found.";
        }
        $tab = 'users';
    }

    // Artists
    if ($action === 'update_artist') {
        $artistId   = (int)($_POST['artist_id'] ?? 0);
        $newStatus  = $_POST['status'] ?? '';
        $featured   = !empty($_POST['featured_on_app']);

        $artists = read_json('artists.json');
        $changed = false;

        foreach ($artists as &$a) {
            if ($a['id'] === $artistId) {
                if (isset($artistStatuses[$newStatus])) {
                    $a['status'] = $newStatus;
                }
                $a['featured_on_app'] = $featured;
                $changed = true;
                break;
            }
        }
        unset($a);

        if ($changed) {
            write_json('artists.json', $artists);
            $notice = "Artist #{$artistId} updated.";
        } else {
            $notice = "Artist not found.";
        }
        $tab = 'artists';
    }

    // Invoices
    if ($action === 'update_invoice') {
        $invoiceId  = (int)($_POST['invoice_id'] ?? 0);
        $newStatus  = $_POST['status'] ?? '';

        $invoices = read_json('invoices.json');
        $changed  = false;

        foreach ($invoices as &$inv) {
            if ($inv['id'] === $invoiceId) {
                if (isset($invoiceStatuses[$newStatus])) {
                    $inv['status'] = $newStatus;
                    $inv['updated_at'] = date('Y-m-d H:i:s');
                }
                $changed = true;
                break;
            }
        }
        unset($inv);

        if ($changed) {
            write_json('invoices.json', $invoices);
            $notice = "Invoice #{$invoiceId} updated.";
        } else {
            $notice = "Invoice not found.";
        }
        $tab = 'invoices';
    }

    // Add artist manually
    if ($action === 'add_artist') {
        $artistName  = trim($_POST['artist_name'] ?? '');
        $ownerUserId = (int)($_POST['owner_user_id'] ?? 0);
        $tagline     = trim($_POST['tagline'] ?? '');
        $heroBanner  = trim($_POST['hero_banner'] ?? '');
        $profileImg  = trim($_POST['profile_image'] ?? '');

        if ($artistName === '') {
            $notice = "Artist name is required to add a new artist.";
        } else {
            $artists = read_json('artists.json');
            $users   = read_json('users.json');

            // Validate owner_user_id
            $ownerUserIdValid = 0;
            if ($ownerUserId > 0) {
                foreach ($users as $u) {
                    if ($u['id'] === $ownerUserId) {
                        $ownerUserIdValid = $ownerUserId;
                        break;
                    }
                }
            }

            $newArtistId = $artists ? (max(array_column($artists, 'id')) + 1) : 1;
            $slug        = slugify($artistName) . '-' . $newArtistId;

            $artists[] = [
                "id"            => $newArtistId,
                "slug"          => $slug,
                "name"          => $artistName,
                "tagline"       => $tagline ?: "Artist on Londons Energy Radio",
                "quote"         => "",
                "quote_credit"  => "",
                "hero_banner"   => $heroBanner,
                "profile_image" => $profileImg,
                "bio"           => [],
                "socials"       => [
                    "instagram" => "",
                    "youtube"   => ""
                ],
                "tracks"          => [],
                "featured_on_app" => false,
                "status"          => "pending",
                "owner_user_id"   => $ownerUserIdValid ?: null
            ];

            write_json('artists.json', $artists);
            $notice = "Artist '{$artistName}' added.";
        }
        $tab = 'artists';
    }
}

// Load fresh data for display
$users    = read_json('users.json');
$artists  = read_json('artists.json');
$invoices = read_json('invoices.json');

// Helper lookup
$userById    = [];
$userByEmail = [];
foreach ($users as $u) {
    $userById[$u['id']] = $u;
    $userByEmail[strtolower($u['email'])] = $u;
}
$artistById = [];
foreach ($artists as $a) {
    $artistById[$a['id']] = $a;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Admin Dashboard • Londons Energy Artist Portal</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <link rel="stylesheet" href="../assets/css/main.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
  <style>
    .admin-wrapper {
      padding: 2rem 1.2rem 3rem;
      max-width: 1100px;
      margin: 0 auto;
    }
    .admin-tabs {
      display: flex;
      gap: 0.5rem;
      margin-top: 1rem;
      margin-bottom: 1rem;
      flex-wrap: wrap;
    }
    .admin-tab-btn {
      border-radius: 999px;
      border: 1px solid rgba(255,255,255,0.2);
      background: rgba(0,0,0,0.7);
      color: #f5f5f5;
      padding: 0.35rem 0.9rem;
      font-size: 0.78rem;
      text-transform: uppercase;
      letter-spacing: 0.16em;
      cursor: pointer;
    }
    .admin-tab-btn--active {
      border-color: rgba(255,123,24,0.9);
      box-shadow: 0 0 14px rgba(255,123,24,0.9);
      color: #ffb36a;
    }
    .admin-tab-panel {
      display: none;
    }
    .admin-tab-panel.active {
      display: block;
    }
    .badge-status {
      border-radius: 999px;
      padding: 0.1rem 0.45rem;
      font-size: 0.7rem;
      text-transform: uppercase;
      letter-spacing: 0.12em;
      border: 1px solid rgba(255,255,255,0.18);
      display: inline-flex;
      align-items: center;
      gap: 0.25rem;
    }
    .badge-status--pending { color:#ffb36a; border-color:rgba(255,123,24,0.7); }
    .badge-status--approved,
    .badge-status--paid { color:#4ce37b; border-color:rgba(76,227,123,0.8); }
    .badge-status--cancelled,
    .badge-status--refunded,
    .badge-status--expired { color:#ff4b4b; border-color:rgba(255,75,75,0.8); }
    .table-scroll {
      max-height: 420px;
      overflow:auto;
    }
    .dash-table th {
      position: sticky;
      top: 0;
      background: rgba(5,5,7,0.96);
      z-index: 2;
    }
  </style>
  <script src="../assets/js/main.js" defer></script>
</head>
<body class="dark-bg">
  <div class="page-bg"></div>

  <header class="site-header site-header--sticky">
    <div class="site-header-inner">
      <a href="../index.php" class="site-brand">
        <img
          src="https://londonsenergyradio.co.uk/wp-content/uploads/2022/12/Energy-radio-p3h439ea3kdydq5pmbj4z4m1ek89863lffdvjk6qmc.png"
          alt="Londons Energy Radio"
          class="site-logo"
        >
        <div class="site-brand-text">
          <span class="site-brand-top">Londons Energy Radio</span>
          <span class="site-brand-sub">Admin Console</span>
        </div>
      </a>
      <button class="nav-toggle" type="button" aria-controls="primary-nav" aria-expanded="false">
        <span class="sr-only">Menu</span>
        <i class="fa-solid fa-bars"></i>
      </button>

      <nav class="site-nav" id="primary-nav" aria-label="Primary">
        <span class="nav-link" style="pointer-events:none;">
          <i class="fa-solid fa-user-shield"></i>
          Admin
        </span>
        <a href="../index.php" class="nav-link">Back to Portal</a>
        <a href="../logout.php" class="nav-link nav-link--outline">
          <i class="fa-solid fa-right-from-bracket"></i>
          Logout
        </a>
      </nav>
    </div>
  </header>

  <div class="nav-overlay" data-nav-close></div>

  <main class="admin-wrapper">
    <div class="dash-header">
      <div>
        <h1 class="section-heading">Admin Dashboard</h1>
        <p class="section-subtitle">
          Manage users, artist submissions and invoices for Londons Energy Artist Portal.
        </p>
      </div>
      <div style="text-align:right;font-size:0.8rem;color:var(--text-muted);">
        <div>Users: <?= count($users) ?></div>
        <div>Artists: <?= count($artists) ?></div>
        <div>Invoices: <?= count($invoices) ?></div>
      </div>
    </div>

    <?php if ($notice): ?>
      <div class="dash-card" style="margin-top:1rem;border-color:rgba(255,123,24,0.8);">
        <p style="margin:0;font-size:0.85rem;"><?= htmlspecialchars($notice) ?></p>
      </div>
    <?php endif; ?>

    <div class="admin-tabs">
      <button class="admin-tab-btn <?= $tab==='artists'?'admin-tab-btn--active':'' ?>" data-tab="artists">
        <i class="fa-solid fa-user-music"></i> Artists / Submissions
      </button>
      <button class="admin-tab-btn <?= $tab==='invoices'?'admin-tab-btn--active':'' ?>" data-tab="invoices">
        <i class="fa-solid fa-file-invoice-dollar"></i> Orders / Invoices
      </button>
      <button class="admin-tab-btn <?= $tab==='users'?'admin-tab-btn--active':'' ?>" data-tab="users">
        <i class="fa-solid fa-users-gear"></i> Users
      </button>
    </div>

    <!-- ARTISTS TAB -->
    <section class="admin-tab-panel <?= $tab==='artists'?'active':'' ?>" id="tab-artists">
      <div class="dash-card">
        <h2>Artist Submissions</h2>
        <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0;">
          Update submission status (<em>Pending / On Hold / Approved / Featured / Expired / Cancelled</em>)
          and toggle “Featured on our App”.
        </p>
        <div class="table-scroll" style="margin-top:0.8rem;">
          <table class="dash-table">
            <thead>
              <tr>
                <th>ID</th>
                <th>Artist</th>
                <th>Owner</th>
                <th>Status</th>
                <th>Featured App</th>
                <th>Profile</th>
                <th>Save</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($artists as $artist): ?>
                <?php
                  $owner = null;
                  if (!empty($artist['owner_user_id']) && isset($userById[$artist['owner_user_id']])) {
                      $owner = $userById[$artist['owner_user_id']];
                  }
                  $status = $artist['status'] ?? 'pending';
                  $badgeClass = 'badge-status--pending';
                  if (in_array($status, ['approved','featured'], true)) $badgeClass = 'badge-status--approved';
                  if (in_array($status, ['cancelled','expired'], true)) $badgeClass = 'badge-status--cancelled';
                ?>
                <tr>
                  <td><?= (int)$artist['id'] ?></td>
                  <td>
                    <strong><?= htmlspecialchars($artist['name']) ?></strong><br>
                    <span style="font-size:0.75rem;color:var(--text-muted);">
                      <?= htmlspecialchars($artist['tagline'] ?? '') ?>
                    </span>
                  </td>
                  <td>
                    <?php if ($owner): ?>
                      <span style="font-size:0.8rem;"><?= htmlspecialchars($owner['name']) ?></span><br>
                      <span style="font-size:0.75rem;color:var(--text-muted);">
                        <?= htmlspecialchars($owner['email']) ?>
                      </span>
                    <?php else: ?>
                      <span style="font-size:0.75rem;color:var(--text-muted);">Unlinked</span>
                    <?php endif; ?>
                  </td>
                  <td>
                    <span class="badge-status <?= $badgeClass ?>">
                      <i class="fa-solid fa-circle"></i>
                      <?= htmlspecialchars($artistStatuses[$status] ?? ucfirst($status)) ?>
                    </span>
                    <form method="post" style="margin-top:0.25rem;">
                      <input type="hidden" name="action" value="update_artist">
                      <input type="hidden" name="artist_id" value="<?= (int)$artist['id'] ?>">
                      <select name="status" style="font-size:0.8rem;width:100%;">
                        <?php foreach ($artistStatuses as $key=>$label): ?>
                          <option value="<?= $key ?>" <?= $key===$status?'selected':'' ?>>
                            <?= $label ?>
                          </option>
                        <?php endforeach; ?>
                      </select>
                  </td>
                  <td style="text-align:center;">
                      <label style="font-size:0.75rem;display:inline-flex;align-items:center;gap:0.3rem;">
                        <input type="checkbox" name="featured_on_app" value="1"
                          <?= !empty($artist['featured_on_app']) ? 'checked' : '' ?>>
                        On App
                      </label>
                  </td>
                  <td>
                    <a href="../artist.php?artist=<?= urlencode($artist['slug']) ?>" target="_blank" class="btn-small">
                      View Profile
                    </a>
                  </td>
                  <td>
                      <button type="submit" class="btn-small">
                        Save
                      </button>
                    </form>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>

      <!-- Quick add artist -->
      <div class="dash-card">
        <h2>Add Artist Manually</h2>
        <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0;">
          Use this if you want to create a profile for an artist without them using the submission form.
          You can paste existing banner/profile URLs and link to a user account if needed.
        </p>
        <form method="post">
          <input type="hidden" name="action" value="add_artist">
          <label>Artist Name *</label>
          <input type="text" name="artist_name" required>

          <label>Link to User (optional)</label>
          <select name="owner_user_id">
            <option value="">-- No linked user --</option>
            <?php foreach ($users as $u): ?>
              <option value="<?= (int)$u['id'] ?>">
                #<?= (int)$u['id'] ?> • <?= htmlspecialchars($u['name']) ?> (<?= htmlspecialchars($u['email']) ?>)
              </option>
            <?php endforeach; ?>
          </select>

          <label>Tagline (optional)</label>
          <input type="text" name="tagline" placeholder="Genre / style etc.">

          <label>Banner Image URL (optional)</label>
          <input type="text" name="hero_banner" placeholder="https://...">

          <label>Profile Image URL (optional)</label>
          <input type="text" name="profile_image" placeholder="https://...">

          <div style="margin-top:0.8rem;">
            <button type="submit" class="btn-primary">
              <i class="fa-solid fa-plus"></i>
              Add Artist
            </button>
          </div>
        </form>
      </div>
    </section>

    <!-- INVOICES TAB -->
    <section class="admin-tab-panel <?= $tab==='invoices'?'active':'' ?>" id="tab-invoices">
      <div class="dash-card">
        <h2>Orders & Invoices</h2>
        <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0;">
          Update invoice status (<em>Pending / Paid / Cancelled / Refunded</em>).
          This controls what artists see in their invoice timeline.
        </p>
        <div class="table-scroll" style="margin-top:0.8rem;">
          <table class="dash-table">
            <thead>
              <tr>
                <th>ID</th>
                <th>User</th>
                <th>Artist</th>
                <th>Package</th>
                <th>Amount</th>
                <th>Status</th>
                <th>Payment Link</th>
                <th>Updated</th>
                <th>Save</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($invoices as $inv): ?>
                <?php
                  $status = $inv['status'] ?? 'pending';
                  $badgeClass = 'badge-status--pending';
                  if ($status === 'paid') $badgeClass = 'badge-status--paid';
                  if (in_array($status, ['cancelled','refunded'], true)) $badgeClass = 'badge-status--cancelled';

                  $user = isset($userById[$inv['user_id']]) ? $userById[$inv['user_id']] : null;
                  $artist = isset($artistById[$inv['artist_id']]) ? $artistById[$inv['artist_id']] : null;
                ?>
                <tr>
                  <td><?= (int)$inv['id'] ?></td>
                  <td>
                    <?php if ($user): ?>
                      <strong><?= htmlspecialchars($user['name']) ?></strong><br>
                      <span style="font-size:0.75rem;color:var(--text-muted);">
                        <?= htmlspecialchars($user['email']) ?>
                      </span>
                    <?php else: ?>
                      <span style="font-size:0.75rem;color:var(--text-muted);">Unknown user</span>
                    <?php endif; ?>
                  </td>
                  <td>
                    <?php if ($artist): ?>
                      <?= htmlspecialchars($artist['name']) ?>
                    <?php else: ?>
                      <span style="font-size:0.75rem;color:var(--text-muted);">Unknown artist</span>
                    <?php endif; ?>
                  </td>
                  <td>
                    <?= htmlspecialchars($inv['package_name'] ?? '') ?>
                    <?php if (!empty($inv['addons'])): ?>
                      <br>
                      <span style="font-size:0.75rem;color:var(--text-muted);">
                        Add-ons: <?= htmlspecialchars(implode(', ', $inv['addons'])) ?>
                      </span>
                    <?php endif; ?>
                  </td>
                  <td>£<?= number_format((float)$inv['amount'], 2) ?></td>
                  <td>
                    <span class="badge-status <?= $badgeClass ?>">
                      <i class="fa-solid fa-circle"></i>
                      <?= htmlspecialchars($invoiceStatuses[$status] ?? ucfirst($status)) ?>
                    </span>
                    <form method="post" style="margin-top:0.25rem;">
                      <input type="hidden" name="action" value="update_invoice">
                      <input type="hidden" name="invoice_id" value="<?= (int)$inv['id'] ?>">
                      <select name="status" style="font-size:0.8rem;">
                        <?php foreach ($invoiceStatuses as $key=>$label): ?>
                          <option value="<?= $key ?>" <?= $key===$status?'selected':'' ?>>
                            <?= $label ?>
                          </option>
                        <?php endforeach; ?>
                      </select>
                  </td>
                  <td>
                    <?php if (!empty($inv['payment_url'])): ?>
                      <a href="<?= htmlspecialchars($inv['payment_url']) ?>" target="_blank" class="btn-small">
                        Pay Link
                      </a>
                    <?php endif; ?>
                  </td>
                  <td style="font-size:0.75rem;color:var(--text-muted);">
                    <?= htmlspecialchars($inv['updated_at'] ?? $inv['created_at'] ?? '') ?>
                  </td>
                  <td>
                      <button type="submit" class="btn-small">Save</button>
                    </form>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>
    </section>

    <!-- USERS TAB -->
    <section class="admin-tab-panel <?= $tab==='users'?'active':'' ?>" id="tab-users">
      <div class="dash-card">
        <h2>Users</h2>
        <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0;">
          Manage account status (<em>Pending / Active / Suspended</em>) and roles (Artist / Admin).
        </p>
        <div class="table-scroll" style="margin-top:0.8rem;">
          <table class="dash-table">
            <thead>
              <tr>
                <th>ID</th>
                <th>Name / Email</th>
                <th>Role</th>
                <th>Status</th>
                <th>Linked Artist</th>
                <th>Save</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($users as $u): ?>
                <?php
                  $status = $u['status'] ?? 'pending';
                  $badgeClass = 'badge-status--pending';
                  if ($status === 'active') $badgeClass = 'badge-status--approved';
                  if ($status === 'suspended') $badgeClass = 'badge-status--cancelled';

                  $linkedArtist = null;
                  if (!empty($u['artist_slug'])) {
                      foreach ($artists as $a) {
                          if ($a['slug'] === $u['artist_slug']) {
                              $linkedArtist = $a;
                              break;
                          }
                      }
                  }
                ?>
                <tr>
                  <td><?= (int)$u['id'] ?></td>
                  <td>
                    <strong><?= htmlspecialchars($u['name']) ?></strong><br>
                    <span style="font-size:0.75rem;color:var(--text-muted);">
                      <?= htmlspecialchars($u['email']) ?>
                    </span>
                  </td>
                  <td>
                    <form method="post">
                      <input type="hidden" name="action" value="update_user">
                      <input type="hidden" name="user_id" value="<?= (int)$u['id'] ?>">
                      <select name="role" style="font-size:0.8rem;width:100%;">
                        <option value="artist" <?= ($u['role'] ?? 'artist')==='artist'?'selected':'' ?>>
                          Artist
                        </option>
                        <option value="admin" <?= ($u['role'] ?? 'artist')==='admin'?'selected':'' ?>>
                          Admin
                        </option>
                      </select>
                  </td>
                  <td>
                    <span class="badge-status <?= $badgeClass ?>">
                      <i class="fa-solid fa-circle"></i>
                      <?= htmlspecialchars($userStatuses[$status] ?? ucfirst($status)) ?>
                    </span>
                    <select name="status" style="font-size:0.8rem;width:100%;margin-top:0.2rem;">
                      <?php foreach ($userStatuses as $key=>$label): ?>
                        <option value="<?= $key ?>" <?= $key===$status?'selected':'' ?>>
                          <?= $label ?>
                        </option>
                      <?php endforeach; ?>
                    </select>
                  </td>
                  <td>
                    <?php if ($linkedArtist): ?>
                      <a href="../artist.php?artist=<?= urlencode($linkedArtist['slug']) ?>" target="_blank" class="btn-small">
                        <?= htmlspecialchars($linkedArtist['name']) ?>
                      </a>
                    <?php else: ?>
                      <span style="font-size:0.75rem;color:var(--text-muted);">None</span>
                    <?php endif; ?>
                  </td>
                  <td>
                      <button type="submit" class="btn-small">Save</button>
                    </form>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>
    </section>
  </main>

  <footer class="site-footer">
    KEEPING IT ENERGY • LONDONS ENERGY RADIO
  </footer>

  <script>
    // Tab switching
    const tabButtons = document.querySelectorAll('.admin-tab-btn');
    const panels     = document.querySelectorAll('.admin-tab-panel');

    tabButtons.forEach(btn => {
      btn.addEventListener('click', () => {
        const target = btn.getAttribute('data-tab');

        tabButtons.forEach(b => b.classList.remove('admin-tab-btn--active'));
        btn.classList.add('admin-tab-btn--active');

        panels.forEach(p => {
          if (p.id === 'tab-' + target) {
            p.classList.add('active');
          } else {
            p.classList.remove('active');
          }
        });

        const url = new URL(window.location.href);
        url.searchParams.set('tab', target);
        window.history.replaceState({}, '', url);
      });
    });
  </script>
</body>
</html>
