<?php
require __DIR__ . '/../includes/config.php';
require __DIR__ . '/../includes/functions.php';

if (empty($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'artist') {
    header("Location: ../login.php");
    exit;
}

$userId = (int)$_SESSION['user_id'];

// Load data
$users    = read_json('users.json');
$artists  = read_json('artists.json');
$invoices = read_json('invoices.json');

// Current user
$currentUser = null;
foreach ($users as $u) {
    if ($u['id'] === $userId) {
        $currentUser = $u;
        break;
    }
}
if (!$currentUser) {
    header("Location: ../login.php");
    exit;
}

// Find artist(s) owned by this user
$myArtists = [];
foreach ($artists as $a) {
    if (($a['owner_user_id'] ?? 0) === $userId) {
        $myArtists[] = $a;
    }
}

// Filter invoices for this user
$myInvoices = [];
foreach ($invoices as $inv) {
    if (($inv['user_id'] ?? 0) === $userId) {
        $myInvoices[] = $inv;
    }
}

// Sort invoices by created_at desc
usort($myInvoices, function($a, $b) {
    return strcmp($b['created_at'] ?? '', $a['created_at'] ?? '');
});

$artistStatusLabels = [
    'pending'   => 'Pending',
    'on_hold'   => 'On Hold',
    'approved'  => 'Approved',
    'featured'  => 'Featured',
    'expired'   => 'Expired',
    'cancelled' => 'Cancelled',
];

$invoiceStatusLabels = [
    'pending'   => 'Pending',
    'paid'      => 'Paid',
    'cancelled' => 'Cancelled',
    'refunded'  => 'Refunded',
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Artist Dashboard • Londons Energy</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <link rel="stylesheet" href="../assets/css/main.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
  <style>
    .artist-dash-wrapper {
      padding: 2rem 1.2rem 3rem;
      max-width: 1100px;
      margin: 0 auto;
    }
    .status-select-readonly {
      width: 100%;
      font-size: 0.8rem;
      background: rgba(0,0,0,0.7);
      border-radius: 0.8rem;
      border: 1px dashed rgba(255,255,255,0.25);
      color: #f5f5f5;
      padding: 0.4rem 0.6rem;
    }
    .status-select-readonly:disabled {
      opacity: 0.7;
    }
    .file-list {
      font-size: 0.8rem;
      color: var(--text-muted);
    }
    .file-list a {
      color: var(--orange-soft);
      text-decoration: none;
    }
    .file-list a:hover {
      text-decoration: underline;
    }
  </style>
  <script src="../assets/js/main.js" defer></script>
</head>
<body class="dark-bg">
  <div class="page-bg"></div>

  <header class="site-header site-header--sticky">
    <div class="site-header-inner">
      <a href="../index.php" class="site-brand">
        <img
          src="https://londonsenergyradio.co.uk/wp-content/uploads/2022/12/Energy-radio-p3h439ea3kdydq5pmbj4z4m1ek89863lffdvjk6qmc.png"
          alt="Londons Energy Radio"
          class="site-logo"
        >
        <div class="site-brand-text">
          <span class="site-brand-top">Londons Energy Radio</span>
          <span class="site-brand-sub">Artist Dashboard</span>
        </div>
      </a>
      <button class="nav-toggle" type="button" aria-controls="primary-nav" aria-expanded="false">
        <span class="sr-only">Menu</span>
        <i class="fa-solid fa-bars"></i>
      </button>

      <nav class="site-nav" id="primary-nav" aria-label="Primary">
        <a href="../index.php" class="nav-link">Portal</a>
        <span class="nav-link" style="pointer-events:none;">
          <i class="fa-solid fa-user"></i>
          <?= htmlspecialchars($currentUser['name']) ?>
        </span>
        <a href="../logout.php" class="nav-link nav-link--outline">
          <i class="fa-solid fa-right-from-bracket"></i>
          Logout
        </a>
      </nav>
    </div>
  </header>

  <div class="nav-overlay" data-nav-close></div>

  <main class="artist-dash-wrapper">
    <div class="dash-header">
      <div>
        <h1 class="section-heading">Your Artist Account</h1>
        <p class="section-subtitle">
          View your submission status, uploaded media and invoice timeline for Londons Energy Radio.
        </p>
      </div>
      <div style="text-align:right;font-size:0.8rem;color:var(--text-muted);">
        <div>Linked Artists: <?= count($myArtists) ?></div>
        <div>Invoices: <?= count($myInvoices) ?></div>
      </div>
    </div>

    <?php if (!$myArtists): ?>
      <div class="dash-card" style="margin-top:1.2rem;">
        <h2>No artist profile found</h2>
        <p style="font-size:0.85rem;color:var(--text-muted);">
          We couldn’t find an artist profile linked to this account yet.  
          If you’ve just submitted, your profile may still be under manual review.
        </p>
        <div style="margin-top:0.7rem;">
          <a href="../submit-artist.php" class="btn-primary">
            Submit Artist Profile
          </a>
        </div>
      </div>
    <?php else: ?>

      <?php foreach ($myArtists as $artist): ?>
        <?php
          $status = $artist['status'] ?? 'pending';
          $statusLabel = $artistStatusLabels[$status] ?? ucfirst($status);
          $statusBadgeClass = 'badge-status--pending';
          if (in_array($status, ['approved','featured'], true)) $statusBadgeClass = 'badge-status--approved';
          if (in_array($status, ['cancelled','expired'], true)) $statusBadgeClass = 'badge-status--cancelled';

          $tracks = $artist['tracks'] ?? [];
        ?>
        <div class="dash-card" style="margin-top:1.2rem;">
          <h2><?= htmlspecialchars($artist['name']) ?></h2>
          <p style="font-size:0.85rem;color:var(--text-muted);margin-top:0.1rem;">
            <?= htmlspecialchars($artist['tagline'] ?? '') ?>
          </p>

          <div class="dash-grid" style="margin-top:1rem;">
            <!-- LEFT: Submission status & links -->
            <div>
              <h3 style="font-size:0.85rem;text-transform:uppercase;letter-spacing:0.14em;color:var(--orange-soft);margin-top:0;">
                Submission Status
              </h3>
              <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0.2rem;">
                This is controlled by the Londons Energy admin team and reflects where you are in the pipeline.
              </p>

              <div style="margin-top:0.4rem;">
                <span class="badge-status <?= $statusBadgeClass ?>">
                  <i class="fa-solid fa-circle"></i>
                  <?= $statusLabel ?>
                </span>
              </div>

              <div style="margin-top:0.5rem;">
                <select class="status-select-readonly" disabled>
                  <?php foreach ($artistStatusLabels as $key=>$label): ?>
                    <option value="<?= $key ?>" <?= $key===$status?'selected':'' ?>>
                      <?= $label ?>
                    </option>
                  <?php endforeach; ?>
                </select>
                <p style="font-size:0.75rem;color:var(--text-muted);margin-top:0.25rem;">
                  Read-only: if something doesn’t look right, contact the station team.
                </p>
              </div>

              <div style="margin-top:0.8rem;">
                <a href="../artist.php?artist=<?= urlencode($artist['slug']) ?>" target="_blank" class="btn-secondary">
                  View Public Profile
                </a>
              </div>

              <div style="margin-top:0.8rem;font-size:0.8rem;color:var(--text-muted);">
                <strong>Featured on our App:</strong>
                <?php if (!empty($artist['featured_on_app'])): ?>
                  <span style="color:#4ce37b;">Yes</span>
                <?php else: ?>
                  <span>No (may be upgraded by the station)</span>
                <?php endif; ?>
              </div>
            </div>

            <!-- RIGHT: Uploaded files & tracks -->
            <div>
              <h3 style="font-size:0.85rem;text-transform:uppercase;letter-spacing:0.14em;color:var(--orange-soft);margin-top:0;">
                Uploaded Media
              </h3>
              <ul class="file-list">
                <li>
                  <strong>Banner Image:</strong>
                  <?php if (!empty($artist['hero_banner'])): ?>
                    <a href="../<?= ltrim($artist['hero_banner'], '/') ?>" target="_blank">
                      <?= htmlspecialchars($artist['hero_banner']) ?>
                    </a>
                  <?php else: ?>
                    <span>Not set</span>
                  <?php endif; ?>
                </li>
                <li>
                  <strong>Profile Image:</strong>
                  <?php if (!empty($artist['profile_image'])): ?>
                    <a href="../<?= ltrim($artist['profile_image'], '/') ?>" target="_blank">
                      <?= htmlspecialchars($artist['profile_image']) ?>
                    </a>
                  <?php else: ?>
                    <span>Not set</span>
                  <?php endif; ?>
                </li>
              </ul>

              <h4 style="font-size:0.8rem;margin-top:0.8rem;">Tracks</h4>
              <?php if (!$tracks): ?>
                <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0.2rem;">
                  No tracks found yet. Your initial submission may still be processing.
                </p>
              <?php else: ?>
                <ul class="file-list">
                  <?php foreach ($tracks as $index=>$t): ?>
                    <li style="margin-bottom:0.45rem;">
                      <strong><?= htmlspecialchars($t['title'] ?? 'Untitled') ?></strong>
                      <?php if (!empty($t['subtitle'])): ?>
                        <span style="font-size:0.78rem;color:var(--text-muted);">
                          <?= htmlspecialchars($t['subtitle']) ?>
                        </span>
                      <?php endif; ?>
                      <?php if (!empty($t['is_featured'])): ?>
                        <span style="font-size:0.7rem;margin-left:0.35rem;color:#ffb36a;">
                          <i class="fa-solid fa-star"></i> Featured
                        </span>
                      <?php endif; ?>
                      <br>
                      <?php if (!empty($t['audio_url'])): ?>
                        <span>Audio:</span>
                        <a href="../<?= ltrim($t['audio_url'], '/') ?>" target="_blank">
                          <?= htmlspecialchars($t['audio_url']) ?>
                        </a>
                        <br>
                      <?php endif; ?>
                      <?php if (!empty($t['artwork'])): ?>
                        <span>Artwork:</span>
                        <a href="../<?= ltrim($t['artwork'], '/') ?>" target="_blank">
                          <?= htmlspecialchars($t['artwork']) ?>
                        </a>
                      <?php endif; ?>
                    </li>
                  <?php endforeach; ?>
                </ul>
              <?php endif; ?>

              <p style="font-size:0.75rem;color:var(--text-muted);margin-top:0.6rem;">
                Need to update a track or artwork? Contact the team or request an update via your usual channel
                until we enable full self-service editing here.
              </p>
            </div>
          </div>
        </div>
      <?php endforeach; ?>

    <?php endif; ?>

    <!-- INVOICE TIMELINE -->
    <div class="dash-card" style="margin-top:1.5rem;">
      <h2>Your Orders & Invoices</h2>
      <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0;">
        These are the invoices generated for your submissions and promotions. Status is updated by the station
        team once payment is confirmed.
      </p>

      <?php if (!$myInvoices): ?>
        <p style="font-size:0.85rem;color:var(--text-muted);margin-top:0.8rem;">
          No invoices found yet. Once you submit and an invoice is created, it will appear here.
        </p>
      <?php else: ?>
        <div class="table-scroll" style="margin-top:0.8rem;max-height:360px;">
          <table class="dash-table">
            <thead>
              <tr>
                <th>#</th>
                <th>Created</th>
                <th>Package</th>
                <th>Add-ons</th>
                <th>Amount</th>
                <th>Status</th>
                <th>Pay Link</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($myInvoices as $inv): ?>
                <?php
                  $st     = $inv['status'] ?? 'pending';
                  $label  = $invoiceStatusLabels[$st] ?? ucfirst($st);
                  $badgeClass = 'badge-status--pending';
                  if ($st === 'paid') $badgeClass = 'badge-status--paid';
                  if (in_array($st, ['cancelled','refunded'], true)) $badgeClass = 'badge-status--cancelled';
                ?>
                <tr>
                  <td><?= (int)$inv['id'] ?></td>
                  <td style="font-size:0.8rem;">
                    <?= htmlspecialchars($inv['created_at'] ?? '') ?>
                  </td>
                  <td>
                    <?= htmlspecialchars($inv['package_name'] ?? '') ?>
                  </td>
                  <td style="font-size:0.8rem;color:var(--text-muted);">
                    <?php if (!empty($inv['addons'])): ?>
                      <?= htmlspecialchars(implode(', ', $inv['addons'])) ?>
                    <?php else: ?>
                      —
                    <?php endif; ?>
                  </td>
                  <td>£<?= number_format((float)$inv['amount'], 2) ?></td>
                  <td>
                    <span class="badge-status <?= $badgeClass ?>">
                      <i class="fa-solid fa-circle"></i>
                      <?= $label ?>
                    </span>
                  </td>
                  <td>
                    <?php if (!empty($inv['payment_url']) && $st === 'pending'): ?>
                      <a href="<?= htmlspecialchars($inv['payment_url']) ?>" target="_blank" class="btn-small">
                        Pay / Reopen
                      </a>
                    <?php else: ?>
                      <span style="font-size:0.75rem;color:var(--text-muted);">
                        <?php if ($st === 'paid'): ?>
                          Paid
                        <?php else: ?>
                          —
                        <?php endif; ?>
                      </span>
                    <?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </main>

  <footer class="site-footer">
    KEEPING IT ENERGY • LONDONS ENERGY RADIO
  </footer>
</body>
</html>
